<?php

namespace App\Exports;
use Modules\Entertainment\Models\Entertainment;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\PageSetup;
use PhpOffice\PhpSpreadsheet\Style\Border;
use Illuminate\Support\Facades\Auth;

class TVshowExport extends BaseExport
{
    public function __construct($columns, $dateRange = [], $type = 'tvshow')
    {
        parent::__construct($columns, $dateRange, $type, __('movie.tvshows'));
    }

    public function headings(): array
    {
        $modifiedHeadings = [];

        foreach ($this->columns as $column) {
            switch ($column) {
                case 'name':
                    $modifiedHeadings[] = __('movie.lbl_name');
                    break;
                case 'description':
                    $modifiedHeadings[] = __('movie.lbl_description');
                    break;
                case 'status':
                    $modifiedHeadings[] = __('messages.lbl_status');
                    break;
                case 'is_restricted':
                    $modifiedHeadings[] = __('movie.lbl_age_restricted');
                    break;
                case 'movie_access':
                    $modifiedHeadings[] = __('movie.lbl_tv_show') . ' ' . __('movie.access');
                    break;
                case 'language':
                    $modifiedHeadings[] = __('movie.lbl_movie_language');
                    break;
                case 'release_date':
                    $modifiedHeadings[] = __('movie.lbl_release_date');
                    break;
                case 'like_count':
                    $modifiedHeadings[] = __('movie.likes');
                    break;
                case 'watch_count':
                    $modifiedHeadings[] = __('movie.watch');
                    break;
                case 'duration':
                    $modifiedHeadings[] = __('movie.lbl_duration');
                    break;
                case 'IMDb_rating':
                case 'imdb_rating':
                    $modifiedHeadings[] = __('movie.lbl_imdb_rating');
                    break;
                case 'content_rating':
                    $modifiedHeadings[] = __('movie.lbl_content_rating');
                    break;
                default:
                    $modifiedHeadings[] = ucwords(str_replace('_', ' ', $column));
                    break;
            }
        }

        return $modifiedHeadings;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $query = Entertainment::where('type', 'tvshow')
            ->withCount([
                'entertainmentLike' => function ($query) {
                    $query->where('is_like', 1);
                },
                'entertainmentView'
            ])
            ->orderBy('updated_at', 'desc')
            ->get();

        $newQuery = $query->map(function ($row) {
            $selectedData = [];

            foreach ($this->columns as $column) {
                switch ($column) {

                    case 'status':
                        $selectedData[$column] = __('messages.inactive');
                        if ($row[$column]) {
                            $selectedData[$column] = __('messages.active');
                        }
                        break;

                    case 'is_restricted':
                        $selectedData[$column] = 'No';
                        if ($row->is_restricted) {
                            $selectedData[$column] = 'Yes';
                        }
                        break;

                    case 'movie_access':
                        // Capitalize first character of access value
                        $accessValue = $row->movie_access ?? '';
                        $selectedData[$column] = !empty($accessValue) ? ucfirst($accessValue) : '';
                        break;

                    case 'language':
                        // Capitalize first character of language value
                        $languageValue = $row->language ?? '';
                        $selectedData[$column] = !empty($languageValue) ? ucfirst($languageValue) : '';
                        break;

                    case 'release_date':
                        $selectedData[$column] = $row->release_date ? formatDate($row->release_date) : '';
                        break;

                    case 'like_count':
                        $selectedData[$column] = $row->entertainment_like_count > 0 ? $row->entertainment_like_count : '-';
                        break;

                    case 'watch_count':
                        $selectedData[$column] = $row->entertainment_view_count > 0 ? $row->entertainment_view_count : '-';
                        break;

                    default:
                        $selectedData[$column] = $row->{$column} ?? null;
                        break;
                }
            }

            return $selectedData;
        });

        return $newQuery;
    }

    /**
     * Override registerEvents to set proper column widths for status and is_restricted
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet;
                $worksheet = $sheet->getDelegate();

                // Get all the parent class properties and methods we need
                $generatedBy = Auth::user()->first_name . ' ' . Auth::user()->last_name ?? 'System';
                $generatedAt = now()->format('d M Y, h:i A');
                $lastColumn = $this->getLastColumn();

                // Set report info (same as parent)
                $sheet->mergeCells("A1:{$lastColumn}1");
                $sheet->setCellValue('A1', $this->reportName);

                $currentRow = 2;
                if (!empty($this->type)) {
                    $sheet->mergeCells("A{$currentRow}:{$lastColumn}{$currentRow}");
                    $sheet->setCellValue("A{$currentRow}", 'Type: ' . ucfirst($this->type));
                    $currentRow++;
                }

                $sheet->mergeCells("A{$currentRow}:{$lastColumn}{$currentRow}");
                $sheet->setCellValue("A{$currentRow}", 'Generated By: ' . $generatedBy);
                $currentRow++;

                $sheet->mergeCells("A{$currentRow}:{$lastColumn}{$currentRow}");
                $sheet->setCellValue("A{$currentRow}", 'Generated On: ' . $generatedAt);

                // Apply styles (same as parent)
                $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
                $sheet->getStyle("A1:A{$currentRow}")->getAlignment()->setHorizontal('left');
                $headingsRow = !empty($this->type) ? 5 : 4;
                $sheet->getStyle("A{$headingsRow}:{$lastColumn}{$headingsRow}")->getFont()->setBold(true);

                // Remove borders from report info section (same as parent)
                $sheet->getStyle("A1:{$lastColumn}{$currentRow}")->applyFromArray([
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => Border::BORDER_NONE,
                        ],
                    ],
                ]);

                // Page setup
                $pageSetup = $worksheet->getPageSetup();
                $pageSetup->setPaperSize(PageSetup::PAPERSIZE_A4)
                    ->setOrientation(count($this->columns) > 3 ? PageSetup::ORIENTATION_LANDSCAPE : PageSetup::ORIENTATION_PORTRAIT)
                    ->setHorizontalCentered(true);

                // Use scaling instead of fit-to-width to ensure all columns are visible
                // This prevents columns from being cut off in PDF
                $columnCount = count($this->columns);
                if ($columnCount >= 10) {
                    // 10+ columns - use smaller scale to fit all including Status
                    $pageSetup->setFitToWidth(false)
                        ->setFitToHeight(false)
                        ->setScale(70);
                } elseif ($columnCount > 8) {
                    // 9 columns - use moderate scale
                    $pageSetup->setFitToWidth(false)
                        ->setFitToHeight(false)
                        ->setScale(75);
                } elseif ($columnCount > 6) {
                    // 7-8 columns - use larger scale
                    $pageSetup->setFitToWidth(false)
                        ->setFitToHeight(false)
                        ->setScale(85);
                } else {
                    // Fewer columns - use fit-to-width
                    $pageSetup->setFitToWidth(1)
                        ->setFitToHeight(0)
                        ->setScale(100);
                }

                // Very tight margins to maximize space for all columns
                $worksheet->getPageMargins()
                    ->setTop(0.15)
                    ->setBottom(0.15)
                    ->setLeft(0.15)
                    ->setRight(0.15);

                // Print area
                $lastRow = $worksheet->getHighestRow();
                $worksheet->getPageSetup()->setPrintArea("A1:{$lastColumn}{$lastRow}");

                // Wrap text and auto-size
                $sheet->getStyle("A{$headingsRow}:{$lastColumn}{$lastRow}")
                    ->getAlignment()
                    ->setWrapText(true)
                    ->setVertical('top');

                $worksheet->getDefaultRowDimension()->setRowHeight(-1);

                // Set optimized column widths for TV Show export
                // Ensure status and is_restricted have proper widths to prevent cutting
                $lastColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($lastColumn);
                $columnWidthMap = [
                    'name' => 25, // Reduced slightly to make room for other columns
                    'movie_access' => 15,
                    'IMDb_rating' => 12,
                    'imdb_rating' => 12,
                    'content_rating' => 18, // Reduced slightly
                    'duration' => 10,
                    'release_date' => 12,
                    'language' => 12, // Reduced slightly
                    'is_restricted' => 15, // Sufficient for "Is Restricted" heading and "Yes"/"No" values
                    'status' => 12, // Increased to 12 to ensure "Active"/"Inactive" values are fully visible
                    'like_count' => 10,
                    'watch_count' => 12,
                ];

                // Map columns to their widths
                for ($col = 1; $col <= $lastColumnIndex; $col++) {
                    $idx = $col - 1;
                    $columnKey = $this->columns[$idx] ?? '';

                    if ($columnKey && array_key_exists($columnKey, $columnWidthMap)) {
                        $worksheet->getColumnDimensionByColumn($col)
                            ->setAutoSize(false)
                            ->setWidth($columnWidthMap[$columnKey]);
                    } else {
                        // Auto-size for columns not in the map
                        $worksheet->getColumnDimensionByColumn($col)->setAutoSize(true);
                    }
                }

                // Auto-height rows
                for ($row = $headingsRow; $row <= $lastRow; $row++) {
                    $worksheet->getRowDimension($row)->setRowHeight(-1);
                }
            },
        ];
    }
}
